<?php
require_once __DIR__ . '/../require/config.php';

$pageTitle = "Galería";
$pageCss = ['galeria.css'];
$pageJs = ['gallery-lightbox.js'];

/**
 * Galería dinámica (desde la BD)
 * - Lee imágenes desde la tabla `gallery` (MySQL) usada por admin/gallery.php
 * - Paginación: 20 imágenes por página
 */
$perPage = 20;
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$offset = ($page - 1) * $perPage;

$images = [];
$total = 0;
$totalPages = 1;
$dbError = null;

try {
  $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
  $pdo = new PDO($dsn, DB_USER, DB_PASS, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);

  $total = (int)$pdo->query("SELECT COUNT(*) FROM gallery")->fetchColumn();
  $totalPages = max(1, (int)ceil($total / $perPage));
  if ($page > $totalPages) {
    $page = $totalPages;
    $offset = ($page - 1) * $perPage;
  }

  $stmt = $pdo->prepare("SELECT id, image_title, image_description, tag, image_path
                         FROM gallery
                         ORDER BY id DESC
                         LIMIT :lim OFFSET :off");
  $stmt->bindValue(':lim', $perPage, PDO::PARAM_INT);
  $stmt->bindValue(':off', $offset, PDO::PARAM_INT);
  $stmt->execute();
  $images = $stmt->fetchAll();
} catch (Throwable $e) {
  // No mostramos detalles en producción; solo un mensaje amable.
  $dbError = "No se pudo cargar la galería en este momento.";
}

include __DIR__ . '/../require/head.php';
include __DIR__ . '/../require/preloader.php';
include __DIR__ . '/../require/header.php';
?>

<main class="container">
  <section class="hero">
    <h1>Galería</h1>
    <p>Momentos bonitos de actividades y proyectos. ✨</p>
  </section>

  <?php if ($dbError): ?>
    <div class="card" style="margin-top: 14px;">
      <p class="muted"><?= htmlspecialchars($dbError, ENT_QUOTES, 'UTF-8') ?></p>
    </div>
  <?php else: ?>

    <?php if (empty($images)): ?>
      <div class="card" style="margin-top: 14px;">
        <p class="muted">Aún no hay fotos en la galería.</p>
      </div>
    <?php else: ?>
      <section class="gallery-grid" aria-label="Galería de fotos">
        <?php foreach ($images as $img): ?>
          <?php
            $path = (string)($img['image_path'] ?? '');
            $src  = BASE_URL . '/' . ltrim($path, '/');
            $title = trim((string)($img['image_title'] ?? ''));
            $desc  = trim((string)($img['image_description'] ?? ''));
            $caption = $title !== '' ? $title : ($desc !== '' ? $desc : 'Foto');
          ?>
          <figure class="gallery-item">
            <button class="gallery-btn" type="button" data-full-src="<?= htmlspecialchars($src, ENT_QUOTES, 'UTF-8') ?>" aria-label="Ver foto">
              <img src="<?= htmlspecialchars($src, ENT_QUOTES, 'UTF-8') ?>"
              alt="<?= htmlspecialchars($caption, ENT_QUOTES, 'UTF-8') ?>"
              loading="lazy">
            </button>
          </figure>
        <?php endforeach; ?>
      </section>

      <?php if ($totalPages > 1): ?>
        <nav class="pagination" aria-label="Paginación">
          <?php
            $base = strtok($_SERVER['REQUEST_URI'], '?');
            $query = $_GET;
          ?>
          <?php if ($page > 1): ?>
            <?php $query['page'] = $page - 1; ?>
            <a class="page-link" href="<?= htmlspecialchars($base . '?' . http_build_query($query), ENT_QUOTES, 'UTF-8') ?>">&larr; Anterior</a>
          <?php else: ?>
            <span class="page-link is-disabled">&larr; Anterior</span>
          <?php endif; ?>

          <span class="page-info">
            Página <?= (int)$page ?> de <?= (int)$totalPages ?>
          </span>

          <?php if ($page < $totalPages): ?>
            <?php $query['page'] = $page + 1; ?>
            <a class="page-link" href="<?= htmlspecialchars($base . '?' . http_build_query($query), ENT_QUOTES, 'UTF-8') ?>">Siguiente &rarr;</a>
          <?php else: ?>
            <span class="page-link is-disabled">Siguiente &rarr;</span>
          <?php endif; ?>
        </nav>
      <?php endif; ?>

    <?php endif; ?>

  <?php endif; ?>

<!-- Lightbox -->
<div class="lightbox" id="lightbox" aria-hidden="true">
  <div class="lightbox__backdrop" data-lb-close></div>
  <div class="lightbox__dialog" role="dialog" aria-modal="true" aria-label="Vista completa de la imagen">
    <button class="lightbox__close" type="button" data-lb-close aria-label="Cerrar">&times;</button>
    <button class="lightbox__nav lightbox__nav--prev" type="button" data-lb-prev aria-label="Anterior">&#10094;</button>
    <figure class="lightbox__figure">
      <img class="lightbox__img" id="lightboxImg" src="" alt="Imagen ampliada" />
    </figure>
    <button class="lightbox__nav lightbox__nav--next" type="button" data-lb-next aria-label="Siguiente">&#10095;</button>
  </div>
</div>

</main>

<?php include __DIR__ . '/../require/footer.php'; ?>
<?php include __DIR__ . '/../require/scripts.php'; ?>
