
(function () {
  const grid = document.querySelector('.gallery-grid');
  if (!grid) return;

  const lightbox = document.getElementById('lightbox');
  const imgEl = document.getElementById('lightboxImg');
  if (!lightbox || !imgEl) return;

  const buttons = Array.from(grid.querySelectorAll('.gallery-btn'));
  const closeEls = Array.from(lightbox.querySelectorAll('[data-lb-close]'));
  const prevBtn = lightbox.querySelector('[data-lb-prev]');
  const nextBtn = lightbox.querySelector('[data-lb-next]');

  let currentIndex = 0;
  let lastFocused = null;

  function setImage(index) {
    if (!buttons.length) return;
    currentIndex = (index + buttons.length) % buttons.length;
    const btn = buttons[currentIndex];
    const src = btn.getAttribute('data-full-src') || '';
    imgEl.src = src;
  }

  function openLightbox(index) {
    lastFocused = document.activeElement;
    setImage(index);
    lightbox.classList.add('is-open');
    lightbox.setAttribute('aria-hidden', 'false');
    document.body.style.overflow = 'hidden';
    // focus close button for accessibility
    const closeBtn = lightbox.querySelector('.lightbox__close');
    if (closeBtn) closeBtn.focus({ preventScroll: true });
  }

  function closeLightbox() {
    lightbox.classList.remove('is-open');
    lightbox.setAttribute('aria-hidden', 'true');
    imgEl.src = '';
    document.body.style.overflow = '';
    if (lastFocused && typeof lastFocused.focus === 'function') {
      lastFocused.focus({ preventScroll: true });
    }
  }

  function next() { setImage(currentIndex + 1); }
  function prev() { setImage(currentIndex - 1); }

  buttons.forEach((btn, idx) => {
    btn.addEventListener('click', () => openLightbox(idx));
    btn.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        openLightbox(idx);
      }
    });
  });

  closeEls.forEach(el => el.addEventListener('click', closeLightbox));
  if (prevBtn) prevBtn.addEventListener('click', prev);
  if (nextBtn) nextBtn.addEventListener('click', next);

  document.addEventListener('keydown', (e) => {
    if (!lightbox.classList.contains('is-open')) return;
    if (e.key === 'Escape') closeLightbox();
    if (e.key === 'ArrowRight') next();
    if (e.key === 'ArrowLeft') prev();
  });

  // Swipe on mobile (and touch devices)
  let touchStartX = 0;
  let touchStartY = 0;
  let touching = false;

  function onTouchStart(e) {
    if (!lightbox.classList.contains('is-open')) return;
    const t = e.touches && e.touches[0];
    if (!t) return;
    touching = true;
    touchStartX = t.clientX;
    touchStartY = t.clientY;
  }

  function onTouchEnd(e) {
    if (!touching || !lightbox.classList.contains('is-open')) return;
    touching = false;
    const t = (e.changedTouches && e.changedTouches[0]) || null;
    if (!t) return;

    const dx = t.clientX - touchStartX;
    const dy = t.clientY - touchStartY;

    // horizontal swipe threshold
    if (Math.abs(dx) > 40 && Math.abs(dx) > Math.abs(dy) * 1.2) {
      if (dx < 0) next();
      else prev();
    }
  }

  lightbox.addEventListener('touchstart', onTouchStart, { passive: true });
  lightbox.addEventListener('touchend', onTouchEnd, { passive: true });

  // Click on image does nothing; click backdrop closes (handled by data-lb-close)
})();
