<?php
require_once __DIR__ . '/../require/config.php';
session_start();

if (empty($_SESSION['admin'])) {
  header('Location: login.php');
  exit;
}
if (($_SESSION['admin_role'] ?? '') !== 'super_admin') {
  header('Location: index.php');
  exit;
}

/* DB */
$dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
$pdo = new PDO($dsn, DB_USER, DB_PASS, [
  PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
  PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
]);

/* CSRF */
if (empty($_SESSION['csrf'])) {
  $_SESSION['csrf'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf'];

$error = '';
$success = '';

if (!empty($_GET['ok'])) {
  $success = 'Cambios guardados correctamente.';
}

/* ACTIONS */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!hash_equals($csrf, $_POST['csrf'] ?? '')) {
    $error = 'Solicitud inválida.';
  } else {
    try {
      $action = $_POST['action'] ?? '';

      /* CREATE */
      if ($action === 'create') {
        $username = trim($_POST['username'] ?? '');
        $password = trim($_POST['password'] ?? '');
        $role     = $_POST['role'] ?? 'moderator';

        if ($username === '' || $password === '') {
          throw new Exception('Datos incompletos.');
        }

        $hash = password_hash($password, PASSWORD_DEFAULT);

        $stmt = $pdo->prepare(
          "INSERT INTO admins (username, password, role) VALUES (?, ?, ?)"
        );
        $stmt->execute([$username, $hash, $role]);

        header("Location: users.php?ok=1");
        exit;
      }

      /* UPDATE */
      if ($action === 'update') {
        $id       = (int)($_POST['id'] ?? 0);
        $username = trim($_POST['username'] ?? '');
        $password = trim($_POST['password'] ?? '');
        $role     = $_POST['role'] ?? 'moderator';

        if ($id <= 0 || $username === '') {
          throw new Exception('Datos inválidos.');
        }

        if ($password !== '') {
          $hash = password_hash($password, PASSWORD_DEFAULT);
          $stmt = $pdo->prepare(
            "UPDATE admins SET username=?, password=?, role=? WHERE id=? LIMIT 1"
          );
          $stmt->execute([$username, $hash, $role, $id]);
        } else {
          $stmt = $pdo->prepare(
            "UPDATE admins SET username=?, role=? WHERE id=? LIMIT 1"
          );
          $stmt->execute([$username, $role, $id]);
        }

        header("Location: users.php?ok=1");
        exit;
      }

      /* DELETE */
      if ($action === 'delete') {
        $id = (int)($_POST['id'] ?? 0);

        $stmt = $pdo->prepare("DELETE FROM admins WHERE id=? LIMIT 1");
        $stmt->execute([$id]);

        header("Location: users.php?ok=1");
        exit;
      }
    } catch (Throwable $e) {
      if (str_contains($e->getMessage(), 'Duplicate')) {
        $error = 'El usuario ya existe.';
      } else {
        $error = 'Error al procesar la acción.';
      }
    }
  }
}

/* LIST */
$admins = $pdo->query(
  "SELECT id, username, role, created_at FROM admins ORDER BY id DESC"
)->fetchAll();

$pageTitle = "Usuarios";
$pageCss = ['admin.css'];
$pageJs  = ['admin.js'];

include __DIR__ . '/../require/head.php';
include __DIR__ . '/../require/preloader.php';
include __DIR__ . '/_layout.php';
?>

<section class="hero">
  <h1>Usuarios</h1>
  <p class="muted">Gestión de usuarios del sistema</p>
</section>

<?php if ($error): ?>
  <div class="notice"><?= htmlspecialchars($error) ?></div>
<?php endif; ?>

<?php if ($success): ?>
  <div class="notice" style="border-color:#2ecc71;">
    <?= htmlspecialchars($success) ?>
  </div>
<?php endif; ?>
<br>
<div class="card users-card">
  <h3>➕ Crear usuario</h3>

  <form method="post" class="form users-form" style="display:flex;gap:12px;flex-wrap:wrap;">
    <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
    <input type="hidden" name="action" value="create">

    <div class="field">
      <label>Usuario</label>
      <input name="username" required>
    </div>

    <div class="field">
      <label>Contraseña</label>
      <input name="password" type="password" required>
    </div>

    <div class="field">
      <label>Rol</label>
      <select name="role">
        <option value="super_admin">super_admin</option>
        <option value="media">media</option>
        <option value="moderator" selected>moderator</option>
        <option value="contact">contact</option>
      </select>
    </div>

    <div class="field" style="align-self:flex-end;">
      <button class="btn primary">Crear</button>
    </div>
  </form>
</div>

<div class="card users-card" style="margin-top:18px;">
  <h3>📋 Usuarios existentes</h3>

  <table class="admin-users-table" style="width:100%;margin-top:10px;">
    <thead>
      <tr>
        <th>ID</th>
        <th>Acciones</th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($admins as $a): ?>
        <tr>
          <td><?= (int)$a['id'] ?></td>
          <td>
            <form method="post" style="display:inline-flex;gap:6px;align-items:center;">
              <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
              <input type="hidden" name="action" value="update">
              <input type="hidden" name="id" value="<?= (int)$a['id'] ?>">

              <input name="username" value="<?= htmlspecialchars($a['username']) ?>" required>
              <input name="password" type="password" placeholder="Nueva contraseña">

              <select name="role">
                <?php foreach (['super_admin','media','moderator','contact'] as $r): ?>
                  <option value="<?= $r ?>" <?= $a['role']===$r?'selected':'' ?>>
                    <?= $r ?>
                  </option>
                <?php endforeach; ?>
              </select>

              <button class="btn">Guardar</button>
            </form>

            <form method="post" style="display:inline-block;margin-left:6px;"
                  onsubmit="return confirm('¿Eliminar este usuario?');">
              <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
              <input type="hidden" name="action" value="delete">
              <input type="hidden" name="id" value="<?= (int)$a['id'] ?>">
              <button class="btn">Eliminar</button>
            </form>
          </td>
        </tr>
      <?php endforeach; ?>
    </tbody>
  </table>
</div>

<?php include __DIR__ . '/../require/scripts.php'; ?>
</div></main></div>
