<?php
// admin/gallery.php (PATCHED: bulk delete fixed + filename clipped in upload UI)

require_once __DIR__ . '/../require/config.php';
session_start();

if (empty($_SESSION['admin'])) { header('Location: login.php'); exit; }

/* DB */
$dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
$pdo = new PDO($dsn, DB_USER, DB_PASS, [
  PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
  PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
]);

/* CSRF */
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(32));
$csrf = $_SESSION['csrf'];

/* Tags */
$allowedTags = ['Ludico','Pintura','Idiomas','Rehabilitacion'];

function normalizeTag(string $s): string {
  $s = trim($s);
  $s = mb_strtolower($s, 'UTF-8');
  $s = strtr($s, ['á'=>'a','é'=>'e','í'=>'i','ó'=>'o','ú'=>'u','ñ'=>'n']);
  return $s;
}
$allowedTagsNorm = [];
foreach ($allowedTags as $t) $allowedTagsNorm[normalizeTag($t)] = $t;

$error = '';
$success = '';
if (!empty($_GET['ok'])) $success = 'Cambios guardados correctamente.';

/* Upload helper */
function saveGalleryImage(array $file): string {
  if (empty($file) || empty($file['name']) || ($file['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_NO_FILE) return '';
  if (($file['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) throw new Exception('Error subiendo imagen.');

  $maxBytes    = 12 * 1024 * 1024;
  $maxDim      = 1600;
  $webpQuality = 72;

  if (($file['size'] ?? 0) > $maxBytes) throw new Exception('Imagen demasiado grande.');

  $uploadDir = __DIR__ . '/../uploads/gallery';
  if (!is_dir($uploadDir) && !mkdir($uploadDir, 0755, true)) throw new Exception('No se pudo crear /uploads/gallery');

  $info = @getimagesize($file['tmp_name']);
  if (!$info) throw new Exception('Archivo no es una imagen válida.');
  $mime = $info['mime'] ?? '';

  if (!function_exists('imagecreatetruecolor') || !function_exists('imagewebp')) {
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $ext = preg_replace('/[^a-z0-9]/', '', $ext);
    if ($ext === '') $ext = 'bin';
    $newName = 'g_' . date('Ymd_His') . '_' . bin2hex(random_bytes(6)) . '.' . $ext;
    $destAbs = $uploadDir . '/' . $newName;
    if (!move_uploaded_file($file['tmp_name'], $destAbs)) throw new Exception('No se pudo guardar la imagen.');
    return 'uploads/gallery/' . $newName;
  }

  $src = null;
  switch ($mime) {
    case 'image/jpeg': $src = @imagecreatefromjpeg($file['tmp_name']); break;
    case 'image/png':  $src = @imagecreatefrompng($file['tmp_name']);  break;
    case 'image/gif':  $src = @imagecreatefromgif($file['tmp_name']);  break;
    case 'image/webp': $src = @imagecreatefromwebp($file['tmp_name']); break;
    default: throw new Exception('Formato no soportado. Usa JPG/PNG/GIF/WebP.');
  }
  if (!$src) throw new Exception('No se pudo leer la imagen.');

  $w = imagesx($src);
  $h = imagesy($src);

  $scale = 1.0;
  $longest = max($w, $h);
  if ($longest > $maxDim) $scale = $maxDim / $longest;

  $newW = max(1, (int)round($w * $scale));
  $newH = max(1, (int)round($h * $scale));

  $dst = imagecreatetruecolor($newW, $newH);

  imagealphablending($dst, false);
  imagesavealpha($dst, true);
  $transparent = imagecolorallocatealpha($dst, 0, 0, 0, 127);
  imagefilledrectangle($dst, 0, 0, $newW, $newH, $transparent);

  if (!imagecopyresampled($dst, $src, 0, 0, 0, 0, $newW, $newH, $w, $h)) {
    imagedestroy($src); imagedestroy($dst);
    throw new Exception('No se pudo procesar la imagen.');
  }

  // WATERMARK PNG
  $wmFile = __DIR__ . '/../assets/images/logo/favicon.png';
  $wmPadding = 20;
  $wmScale = 0.25;

  if (file_exists($wmFile)) {
    $wm = @imagecreatefrompng($wmFile);
    if ($wm) {
      imagealphablending($wm, true);
      imagesavealpha($wm, true);

      $wmW = imagesx($wm);
      $wmH = imagesy($wm);

      $targetW = max(1, (int)round($newW * $wmScale));
      $ratio   = $targetW / max(1, $wmW);
      $targetH = max(1, (int)round($wmH * $ratio));

      $wmResized = imagecreatetruecolor($targetW, $targetH);
      imagealphablending($wmResized, false);
      imagesavealpha($wmResized, true);
      $t2 = imagecolorallocatealpha($wmResized, 0, 0, 0, 127);
      imagefilledrectangle($wmResized, 0, 0, $targetW, $targetH, $t2);

      imagecopyresampled($wmResized, $wm, 0, 0, 0, 0, $targetW, $targetH, $wmW, $wmH);

      $x = max(0, $newW - $targetW - $wmPadding);
      $y = max(0, $newH - $targetH - $wmPadding);

      imagealphablending($dst, true);
      imagesavealpha($dst, true);
      imagecopy($dst, $wmResized, $x, $y, 0, 0, $targetW, $targetH);

      imagedestroy($wm);
      imagedestroy($wmResized);
    }
  }

  $newName = 'g_' . date('Ymd_His') . '_' . bin2hex(random_bytes(6)) . '.webp';
  $destAbs = $uploadDir . '/' . $newName;

  if (!imagewebp($dst, $destAbs, $webpQuality)) {
    imagedestroy($src); imagedestroy($dst);
    throw new Exception('No se pudo guardar la imagen en WebP.');
  }

  imagedestroy($src);
  imagedestroy($dst);

  return 'uploads/gallery/' . $newName;
}

/* ACTIONS */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!hash_equals($csrf, $_POST['csrf'] ?? '')) {
    $error = 'Solicitud inválida.';
  } else {
    try {
      $action = $_POST['action'] ?? '';

      // MULTI CREATE
      if ($action === 'create_multi') {
        if (empty($_FILES['images']) || empty($_FILES['images']['name'])) throw new Exception('No hay imágenes.');

        $names = $_FILES['images']['name'];
        $tmp   = $_FILES['images']['tmp_name'];
        $err   = $_FILES['images']['error'];
        $size  = $_FILES['images']['size'];
        $type  = $_FILES['images']['type'] ?? [];

        $count = is_array($names) ? count($names) : 0;
        if ($count <= 0) throw new Exception('No hay imágenes.');

        for ($i = 0; $i < $count; $i++) {
          if (($err[$i] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_NO_FILE) continue;
          if (($err[$i] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) throw new Exception('Error subiendo una imagen.');

          $title = trim($_POST['image_title'][$i] ?? '');
          $desc  = trim($_POST['image_description'][$i] ?? '');

          $tagRaw = (string)($_POST['tag'][$i] ?? '');
          $tagKey = normalizeTag($tagRaw);
          if (!isset($GLOBALS['allowedTagsNorm'][$tagKey])) throw new Exception('Tag inválido.');
          $tag = $GLOBALS['allowedTagsNorm'][$tagKey];

          $oneFile = [
            'name'     => $names[$i],
            'type'     => $type[$i] ?? '',
            'tmp_name' => $tmp[$i],
            'error'    => $err[$i],
            'size'     => $size[$i],
          ];

          $path = saveGalleryImage($oneFile);
          if ($path === '') continue;

          $stmt = $pdo->prepare(
            "INSERT INTO gallery (image_title, image_description, tag, image_path)
             VALUES (?, ?, ?, ?)"
          );
          $stmt->execute([$title, $desc, $tag, $path]);
        }

        header("Location: gallery.php?ok=1");
        exit;
      }

      // BULK DELETE (FIXED: only deletes selected IDs; never touches single delete)
      if ($action === 'delete_bulk') {
        $ids = $_POST['ids'] ?? [];
        if (!is_array($ids)) $ids = [];
        $ids = array_values(array_filter(array_map('intval', $ids), fn($v) => $v > 0));
        if (!$ids) throw new Exception('Selecciona al menos un item.');

        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $stmt = $pdo->prepare("DELETE FROM gallery WHERE id IN ($placeholders)");
        $stmt->execute($ids);

        header("Location: gallery.php?ok=1");
        exit;
      }

      // UPDATE
      if ($action === 'update') {
        $id    = (int)($_POST['id'] ?? 0);
        $title = trim($_POST['image_title'] ?? '');
        $desc  = trim($_POST['image_description'] ?? '');

        $tagRaw = (string)($_POST['tag'] ?? '');
        $tagKey = normalizeTag($tagRaw);
        if (!isset($allowedTagsNorm[$tagKey])) throw new Exception('Tag inválido.');
        $tag = $allowedTagsNorm[$tagKey];

        if ($id <= 0) throw new Exception('ID inválido.');

        $stmt = $pdo->prepare(
          "UPDATE gallery
           SET image_title=?, image_description=?, tag=?
           WHERE id=? LIMIT 1"
        );
        $stmt->execute([$title, $desc, $tag, $id]);

        header("Location: gallery.php?ok=1");
        exit;
      }

      // INDIVIDUAL DELETE
      if ($action === 'delete') {
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) throw new Exception('ID inválido.');

        $stmt = $pdo->prepare("DELETE FROM gallery WHERE id=? LIMIT 1");
        $stmt->execute([$id]);

        header("Location: gallery.php?ok=1");
        exit;
      }

    } catch (Throwable $e) {
      error_log("Gallery error: " . $e->getMessage());
      $error = $e->getMessage();
    }
  }
}

/* Pagination */
$perPageOptions = [10, 25, 50, 100];
$perPage = (int)($_GET['per_page'] ?? 25);
if (!in_array($perPage, $perPageOptions, true)) $perPage = 25;

$page = (int)($_GET['p'] ?? 1);
if ($page < 1) $page = 1;

$total = (int)$pdo->query("SELECT COUNT(*) FROM gallery")->fetchColumn();
$pages = max(1, (int)ceil($total / $perPage));
if ($page > $pages) $page = $pages;

$offset = ($page - 1) * $perPage;

$stmt = $pdo->prepare(
  "SELECT id, image_title, image_description, tag, image_path
   FROM gallery
   ORDER BY id DESC
   LIMIT :lim OFFSET :off"
);
$stmt->bindValue(':lim', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':off', $offset, PDO::PARAM_INT);
$stmt->execute();
$items = $stmt->fetchAll();

/* Page includes */
$pageTitle = "Galería";
$pageCss = ['admin.css'];
$pageJs  = ['admin.js'];

include __DIR__ . '/../require/head.php';
include __DIR__ . '/../require/preloader.php';
include __DIR__ . '/_layout.php';
?>

<section class="hero">
  <h1>Galería</h1>
  <p class="muted">Subida múltiple + edición compacta + borrado múltiple</p>
</section>

<?php if ($error): ?><div class="notice"><?= htmlspecialchars($error) ?></div><?php endif; ?>
<?php if ($success): ?><div class="notice" style="border-color:#2ecc71;"><?= htmlspecialchars($success) ?></div><?php endif; ?>

<style>
/* Clip long filenames (50px width as requested) */
.fn-clip{
  display:inline-block;
  max-width:50px;
  white-space:nowrap;
  overflow:hidden;
  text-overflow:ellipsis;
  vertical-align:middle;
}
</style>

<!-- MULTI UPLOAD -->
<div class="card users-card">
  <h3>➕ Agregar imágenes (multi)</h3>

  <form method="post" enctype="multipart/form-data" class="form" id="multiForm">
    <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
    <input type="hidden" name="action" value="create_multi">

    <div class="field">
      <label>Seleccionar imágenes</label>
      <input id="multiFiles" type="file" name="images[]" accept=".jpg,.jpeg,.png,.webp" multiple required>
    </div>

    <div id="multiRows" style="margin-top:12px; display:grid; gap:12px;"></div>

    <button class="btn primary" type="submit" style="margin-top:12px;">Agregar todo</button>
  </form>
</div>

<!-- LIST -->
<div class="card users-card" style="margin-top:18px;">
  <div style="display:flex; align-items:center; justify-content:space-between; gap:12px; flex-wrap:wrap;">
    <h3 style="margin:0;">📋 Imágenes (<?= (int)$total ?>)</h3>

    <form method="get" style="display:flex; gap:10px; align-items:center; flex-wrap:wrap;">
      <label class="muted" style="font-size:.9rem;">Filas por página</label>
      <select name="per_page" onchange="this.form.submit()">
        <?php foreach ($perPageOptions as $opt): ?>
          <option value="<?= $opt ?>" <?= $perPage===$opt?'selected':'' ?>><?= $opt ?></option>
        <?php endforeach; ?>
      </select>
      <input type="hidden" name="p" value="1">
    </form>
  </div>

  <!-- IMPORTANT: bulk delete button submits ONLY this form -->
  <form method="post" id="bulkForm" style="margin-top:10px;">
    <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
    <input type="hidden" name="action" value="delete_bulk">

    <div style="display:flex; justify-content:space-between; align-items:center; gap:10px; flex-wrap:wrap; margin:10px 0;">
      <label style="display:flex; align-items:center; gap:8px;">
        <input type="checkbox" id="checkAll">
        <span class="muted">Seleccionar todo</span>
      </label>

      <button class="btn" type="submit"
              onclick="return confirm('¿Eliminar los seleccionados?');"
              title="Eliminar seleccionados"
              style="display:flex; align-items:center; gap:8px;">
        🗑️ <span class="muted" style="font-weight:700;">Eliminar seleccionados</span>
      </button>
    </div>

    <div style="overflow:auto;">
      <table style="width:100%; border-collapse:separate; border-spacing:0 8px;">
        <thead>
          <tr class="muted" style="text-align:left;">
            <th style="width:34px;"></th>
            <th style="width:60px;">Img</th>
            <th style="width:70px;">ID</th>
            <th style="width:160px;">Título</th>
            <th>Descripción</th>
            <th style="width:150px;">Tag</th>
            <th style="width:150px;">Acciones</th>
          </tr>
        </thead>

        <tbody>
          <?php foreach ($items as $it): ?>
            <tr style="height:55px;">
              <td style="padding:8px 6px;">
                <input type="checkbox" name="ids[]" value="<?= (int)$it['id'] ?>" class="rowCheck">
              </td>

              <td style="padding:8px 6px;">
                <?php if (!empty($it['image_path'])): ?>
                  <img src="<?= BASE_URL . '/' . htmlspecialchars($it['image_path']) ?>"
                       alt=""
                       style="width:55px;height:55px;object-fit:cover;border-radius:12px;border:1px solid var(--border);">
                <?php endif; ?>
              </td>

              <td style="padding:8px 6px; font-weight:800;">
                <?= (int)$it['id'] ?>
              </td>

              <td style="padding:8px 6px;">
                <input form="rowForm<?= (int)$it['id'] ?>" name="image_title"
                       value="<?= htmlspecialchars($it['image_title'] ?? '') ?>"
                       placeholder="Título"
                       style="width:150px; padding:8px 10px; border-radius:12px;">
              </td>

              <td style="padding:8px 6px;">
                <input form="rowForm<?= (int)$it['id'] ?>" name="image_description"
                       value="<?= htmlspecialchars($it['image_description'] ?? '') ?>"
                       placeholder="Descripción"
                       style="width:100%; min-width:220px; padding:8px 10px; border-radius:12px;">
              </td>

              <td style="padding:8px 6px;">
                <select form="rowForm<?= (int)$it['id'] ?>" name="tag" required
                        style="width:140px; padding:8px 10px; border-radius:12px;">
                  <?php foreach ($allowedTags as $t): ?>
                    <option value="<?= htmlspecialchars($t) ?>" <?= ($it['tag']===$t?'selected':'') ?>>
                      <?= htmlspecialchars($t) ?>
                    </option>
                  <?php endforeach; ?>
                </select>
              </td>

              <td style="padding:8px 6px; white-space:nowrap;">
                <button class="btn" type="submit" form="rowForm<?= (int)$it['id'] ?>"
                        title="Guardar" style="padding:8px 10px; border-radius:12px;">💾</button>

                <form method="post" style="display:inline-block; margin-left:6px;"
                      onsubmit="return confirm('¿Eliminar esta imagen?');">
                  <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
                  <input type="hidden" name="action" value="delete">
                  <input type="hidden" name="id" value="<?= (int)$it['id'] ?>">
                  <button class="btn" type="submit" title="Eliminar"
                          style="padding:8px 10px; border-radius:12px;">🗑️</button>
                </form>

                <form id="rowForm<?= (int)$it['id'] ?>" method="post" style="display:none;">
                  <input type="hidden" name="csrf" value="<?= htmlspecialchars($csrf) ?>">
                  <input type="hidden" name="action" value="update">
                  <input type="hidden" name="id" value="<?= (int)$it['id'] ?>">
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </form>

  <div style="display:flex; justify-content:space-between; align-items:center; gap:10px; flex-wrap:wrap; margin-top:12px;">
    <div class="muted" style="font-size:.9rem;">Página <?= (int)$page ?> / <?= (int)$pages ?></div>
    <div style="display:flex; gap:8px; flex-wrap:wrap;">
      <?php
        $base = 'gallery.php?per_page=' . urlencode($perPage) . '&p=';
        $prev = max(1, $page - 1);
        $next = min($pages, $page + 1);
      ?>
      <a class="btn" href="<?= $base . $prev ?>" style="padding:8px 12px;">◀</a>
      <a class="btn" href="<?= $base . $next ?>" style="padding:8px 12px;">▶</a>
    </div>
  </div>
</div>

<script>
(function(){
  // Select all checkbox
  const all = document.getElementById('checkAll');
  const checks = () => Array.from(document.querySelectorAll('.rowCheck'));
  if (all) all.addEventListener('change', () => checks().forEach(cb => cb.checked = all.checked));

  // Multi upload rows (clip filename to 50px)
  const input = document.getElementById('multiFiles');
  const rows  = document.getElementById('multiRows');
  const tags  = <?= json_encode($allowedTags) ?>;

  function esc(s){
    return String(s).replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[m]));
  }

  if (input) {
    input.addEventListener('change', () => {
      rows.innerHTML = '';
      const files = Array.from(input.files || []);
      files.forEach((f, i) => {
        const tagOptions = tags.map(t => `<option value="${esc(t)}">${esc(t)}</option>`).join('');
        rows.insertAdjacentHTML('beforeend', `
          <div class="admin-wrap" style="padding:14px;">
            <div style="display:flex; gap:12px; align-items:center; flex-wrap:wrap;">
              <div style="min-width:220px;">
                <strong>Archivo:</strong>
                <span class="fn-clip" title="${esc(f.name)}">${esc(f.name)}</span>
              </div>

              <div class="field">
                <label>Título</label>
                <input name="image_title[${i}]" placeholder="Opcional">
              </div>

              <div class="field" style="min-width:260px; flex:1;">
                <label>Descripción</label>
                <input name="image_description[${i}]" placeholder="Opcional">
              </div>

              <div class="field">
                <label>Tag</label>
                <select name="tag[${i}]" required>
                  ${tagOptions}
                </select>
              </div>
            </div>
          </div>
        `);
      });
    });
  }
})();
</script>

<?php include __DIR__ . '/../require/scripts.php'; ?>
</div></main></div>
